/**
 * CascadeLagerSuche.es: modul mit Utilityfunktionen fr die Suche von Lagern
 * Alle Utilityfunktionen liegen in einem Namespace 'Cascade.Lagersuche'
 *
 **/

// Namespace erzeugen
if ( typeof( Cascade ) == "undefined" )
	Cascade = {};	
if ( !Cascade.Lagersuche )
	Cascade.Lagersuche = {};
	
/**
 * @var Storepfad fr temporres Zwischenspeichern von gefundenen Lagerlisten
 */
 
Cascade.Lagersuche._strTempStorePath = "DLGDATA.TEMP_RESULTS.LAGER_AUSWAHL";

/**
 * testet, ob eine Lagerart ein spezielles Fhrungslager bezeichnet
 *
 * @params string strLagerart - die Lagerart
 * @returns	boolean true - Lagerart ist spezielles Fhrungslager
 *                  false - Lagerart ist kein spezielles Fhrungslager
 */
Cascade.Lagersuche.isSpeziellesFuehrungslager = function( strLagerart ) 
{
	var bFuehrung = false;
	if ( ( strLagerart == 'ID_FUEHRUNG_W2_180' ) || ( strLagerart == 'ID_FUEHRUNG_W2_180_PLUS' ) ||
		 ( strLagerart == 'ID_FUEHRUNG_W2_090' ) || ( strLagerart == 'ID_FUEHRUNG_W3' ) || ( strLagerart == 'ID_FUEHRUNG_W4' ) )
		bFuehrung = true;
		
	return( bFuehrung );
}
	

/**
 * berechnet die Vorgabewerte fr den Werkstoff des Lagers
 *
 * @params string strLagerart - die Lagerart
 *         string strIsolierung - die Isolierung
 *	       string strWerkstoff - die Ausgewaehlte Werkstoffklasse
 *         double dTemperatur - die Betriebstemperatur
 * @returns	object Vorgabewerte
 *                integer wert1
 *                integer wert2
 *                Array aAvailableMaterials
 */
Cascade.Lagersuche.berechneVorgabewerte = function( strLagerart, strIsolierung, strWerkstoff, dTemperatur, oZusatzfunktionen ) 
{
	var nWert1 = "";
	var nWert2 = "";

    /*
        37  S235JRG2	(RSt37-2)	1.0038	0	-10	300
        16  16Mo3	    (15Mo3)	    1.5415	1	250	530
        13  13CrMo4-5	(13CrMo44)	1.7335	2	350	570
        10  10CrMo9-10	(10CrMo910)	1.7380	3	500	600
        41  X6CrNiTi18-10		    1.4541	4	-90	550
		43	X5CrNi18-10				1.4301		-90	550
        71  X6CrNiMoTi17-12-2		1.4571	5	-90	550
        91  X10CrMoVNb9-1	(P91)	1.4903	6	520	650
		80  X5CrAlTi31-20	(800H)	1.4958	7	500	900
    */
    /* Typisierung von Schellen/Lagerwerkstoffen in Ferrit / Austenit:
    var Werkstoffe = {
        W_RST       : "37",  F
        W_15MO3     : "16",  F
        W_13CRMO    : "13",  F
        W_10CRMO    : "10",  F
        W_4541      : "41",  A
		W_4301		: "43",  A
        W_4571      : "71",  A
        W_P91       : "91",  A/F
		W_INCOLOY   : "80",  A
    };
    */
    
    var aAvailableMaterials = [];

	// -Schu- bestimme Grenztemp. nach RH-Voreinstellung (VGB od. WIT)
	var GTo = [];
	var GTu = [];
	var GTTyp = toupper(right(getstring("GRENZ_TEMP", "PROJECT.SETTINGS.CASCADE"), 3));
	for (var i = 0; i < 8; i++)
	{
		var strSQLStatement = "select * from GRENZTEMP where nr="+i;
		var numRet = dbselect("CASCADE", strSQLStatement, "GRENZTEMP");
		if ((GTTyp == "VGB") || (GTTyp == "SME"))
		{
			GTu[i] = dbgetdouble("GRENZTEMP", "uvgb");
			GTo[i] = dbgetdouble("GRENZTEMP", "ovgb");
		}
		else
		{
			GTu[i] = dbgetdouble("GRENZTEMP", "uwit");
			GTo[i] = dbgetdouble("GRENZTEMP", "owit");
		}
	}
	// welche Werkstoffe mglich sind, ist abhngig von der Werkstoffklasse und dem eingestellten Temperaturintervall
	if ( strWerkstoff == "ID_FERRIT" )
	{
		if (( dTemperatur >= GTu[0] ) && ( dTemperatur <= (GTo[0] + 20) ))
			aAvailableMaterials.push("37");
		if (( dTemperatur >= GTu[1] ) && ( dTemperatur <= GTo[1] ))
			aAvailableMaterials.push("16");
		if (( dTemperatur >= GTu[2] ) && ( dTemperatur <= GTo[2] ))
			aAvailableMaterials.push("13");
		if (( dTemperatur >= GTu[3] ) && ( dTemperatur <= GTo[3] ))
			aAvailableMaterials.push("10");
	}
	if ( strWerkstoff == "ID_AUSTENIT" )
	{
		if (( dTemperatur >= GTu[4] ) && ( dTemperatur <= GTo[4] ))
        {
			if (( strIsolierung == "ID_UNGEDAEMMT" ) && ( oZusatzfunktionen.bGleitunterlage ))
			// if (( strIsolierung == "ID_UNGEDAEMMT" ) )
				aAvailableMaterials.push("43");
			else
				aAvailableMaterials.push("41");
        }
		if (( dTemperatur >= GTu[5] ) && ( dTemperatur <= GTo[5] ))
			aAvailableMaterials.push("71");
		if (( dTemperatur >= GTu[7] ) && ( dTemperatur <= GTo[7] ))
			aAvailableMaterials.push("80");
	}
//	if ( ( strWerkstoff == "ID_FERRIT" ) || ( strWerkstoff == "ID_AUSTENIT" ) )
	if ( strWerkstoff == "ID_FERRIT" )
	{
		if (( dTemperatur >= GTu[6] ) && ( dTemperatur <= GTo[6] ))
			aAvailableMaterials.push("91");
    }
	
	if ( dTemperatur <= (GTo[0] + 20) )
	{
		// Werte bei Betriebstemperatur < 300 C
		nWert1 = 37;
		nWert2 = 37;
	}
	else if ( dTemperatur <= 450 )
	{
		// Werte bei Betriebstemperatur zwischen 300 C und 450 C
		nWert1 = 16;
		nWert2 = 16;
		
		// spezielle Werte fr vorisoliete Los- oder einfache Fhrungslager
		if ( ( strIsolierung == "ID_VORISOLIERT" ) && ( ( strLagerart == "ID_LOSLAGER" ) || ( strLagerart == "ID_FUEHRUNGSLAGER" ) ) )
		{
			nWert1 = 37;
			nWert2 = 37;
		}
	}
	else if ( dTemperatur <= GTo[1] )
	{
		// Werte bei Betriebstemperatur bis Grenze 16Mo3
		nWert1 = 16;
		nWert2 = 13;
	}
	else if ( dTemperatur <= GTo[2] )
	{
		// Werte bei Betriebstemperatur bis Grenze 13CrMo4-5
		nWert1 = 13;
		nWert2 = 10;
	}
	else if ( dTemperatur <= GTo[3] )
	{
		// Werte bei Betriebstemperatur bis Grenze 10CrMo9-10
		nWert1 = 10;
		nWert2 = 10;
	}
	else if ( dTemperatur <= GTo[6] )
	{
		// Werte bei Betriebstemperatur bis Grenze P91
		nWert1 = 91;
		nWert2 = 91;
	}
	// else fr Betriebstemperatur > 650 C gibt es zur Zeit keine Lager

	return( { wert1: nWert1, wert2: nWert2, AvailableMaterials: aAvailableMaterials } );
}


/**
 * berechnet die fr die Suche relevante Baureihe des Lagers. Diese Baureihe entspricht dem Tabellennamen der Datenbanktabelle, in der
 * die Lagerdaten fr die bei der aktuellen Suche relevanten Lager abgespeichert sind
 *
 * @params string strLagerart - die Lagerart
 *         string strGleitart - die Gleitart
 *		   string strGleitelement - das Gleitelement
 *         string strIsolierung - die Isolierung
 *         object oZusatzfunktionen - Zusatzfunktionen fr das Lager
 *				  boolean bAbhebesicherung - Abhebesicherung gewhlt?
 *				  boolean bHoehenverstellung - Hhenverstellung gewhlt?
 *				  boolean bGleitunterlage - Gleitunterlage gewhlt?
 *				  boolean bKlemmbar - Klemmbar gewhlt?
 *				  boolean bSchraubbar - schraubbar gewhlt?
 *				  boolean bEinschellig - einschellig gewhlt?
 *				  boolean bBuegelausfuehrung - Bgelausfhrung gewhlt? 
 * @returns	string Kennung der relevanten Baureihe
 */
Cascade.Lagersuche.berechneRelevanteBaureihe = function( strLagerart, strGleitart, strGleitelement, strIsolierung, oZusatzfunktionen, strAuswahlKontext )
{
	var strFirstChar = "";
	var strSecondChar = "";
	var strThirdChar = "";
	
	// Baureihenkennung fr Festlager
	if ( ( strLagerart == "ID_FESTLAGER" ) || ( strLagerart == "ID_FESTLAGER_D" ) )
	{
		// erster Buchstabe generell 'F' fr Festlager
		strFirstChar = "F";

		// bestimme zweiten Buchstaben
		if ( oZusatzfunktionen.bHoehenverstellung )
			strSecondChar = "V";
		else
		{
			if ( oZusatzfunktionen.bSchraubbar )
				strSecondChar = "S";
			else strSecondChar = "L";
		}

		// bestimme dritten Buchstaben
		if ( ( strIsolierung == "ID_UNGEDAEMMT" ) || ( strIsolierung == "ID_WAERMEGEDAEMMT" ) )
		{
			if ( strLagerart == "ID_FESTLAGER" )
				strThirdChar = "N";
			else strThirdChar = "D";
		}
		else
		{
			if ( strIsolierung == "ID_VORISOLIERT" )
				strThirdChar = "V";
			else strThirdChar = "K"; 	// Klteisolierung wird zur Zeit nicht im Katalog angeboten
		}
	}
	else // fr Loslager, Fhrungslager und spezielle Fhrungen
	{
		// bestimme ersten Buchstaben
		if ( ( strIsolierung == "ID_UNGEDAEMMT" ) || ( strIsolierung == "ID_WAERMEGEDAEMMT" ) )
			strFirstChar = "L";
		else if ( strIsolierung == "ID_VORISOLIERT" )
			strFirstChar = "I";
		else strFirstChar = "K";			// gibts noch nicht !!

		// bestimme zweiten Buchstaben
		if ( strGleitart == "ID_EINFACH" )
		{
            if ( strAuswahlKontext == "STUETZE_PTFE" )
				strSecondChar = "X";
            else strSecondChar = "S";
        }
		else if ( strGleitart == "ID_REIBUNGSARM" )
		{
			if ( oZusatzfunktionen.bHoehenverstellung )
				strSecondChar = "V";
			else strSecondChar = "K";
		}
		else if ( strGleitart == "ID_GLEITBLECH" )
            strSecondChar = "X";
		else 
            strSecondChar = "X";		// rollend wird im Katalog noch nicht untersttzt

		// bestimme dritten Buchstaben
		if ( strLagerart == "ID_LOSLAGER" )
		{
			if (strGleitelement == "ID_OHNE" )
				strThirdChar = "L";
			else if (strGleitelement == "ID_LGA" )
				strThirdChar = "A";
			else if (strGleitelement == "ID_LGV" )
				strThirdChar = "V";
		}
		else if ( strLagerart == "ID_FUEHRUNGSLAGER" )
			strThirdChar = "G";
		else if ( strLagerart == "ID_FUEHRUNGSLAGER_W" )
			strThirdChar = "W";
		else if ( Cascade.Lagersuche.isSpeziellesFuehrungslager( strLagerart ) )
			strThirdChar = "F";
		// else der einzige Fall, in der die relevante Baureihe nur zwei buchstaben hat
	}

	return( strFirstChar + strSecondChar + strThirdChar );
}


/**
 * berechne den Schlssel fr die Dmmung
 *
 * @params string strIsolierung - die Isolierung
 * @returns	string Dmmungsschlssel
 */
Cascade.Lagersuche.berechneDaemmschluessel = function( strIsolierung )
{
	var strDaemm = "";
	if ( strIsolierung == "ID_UNGEDAEMMT" )
		strDaemm = "ud"
	else if ( strIsolierung == "ID_VORISOLIERT" )
		strDaemm = "wv"
	else if (strIsolierung == "ID_WAERMEGEDAEMMT" )
		strDaemm = "wd"
	else strDaemm = "kd"   // Kltegedaemmt, wird zur Zeit nicht im Katalog untersttzt
	
	return( strDaemm );
}


/**
 * runde einen Belastungswert
 *
 * @params double dBelastung - Wert fr die Belastung
 * @returns	double - die gerundete Belastung
 */
Cascade.Lagersuche.rundeBelastung = function( dBelastung )
{
	var strBelastung = "" + dBelastung;
	var dGerundeteBelastung = dBelastung;
	if ( dBelastung < 10 )
	{
		if ( find( strBelastung, "." ) != -1 )
		{
			strBelastung = strBelastung + "00"
			dGerundeteBelastung = substr( strBelastung, 1, 3 ) - 0.0;
			if ( ( substr( strBelastung, 4, 4 ) - 0.0 ) >= 5 )
				dGerundeteBelastung += 0.1;
		}
	}
	else
	{
		if ( find( strBelastung, "." ) != -1 )
			dGerundeteBelastung = substr( strBelastung, 1, find( strBelastung, "." ) - 1 ) - 0;
	}
	
	return( dGerundeteBelastung );
}


/**
 * berechnet die fr die Suche relevante Baureihe des Lagers. Diese Baureihe entspricht dem Tabellennamen der Datenbanktabelle, in der
 * die Lagerdaten fr die bei der aktuellen Suche relevanten Lager abgespeichert sind
 *
 * @params string strLagerart - die Lagerart
 *         string strGleitart - die Gleitart
 *         string strGleitelement - das Gleitelement
 *         string strIsolierung - die Isolierung
 *         string strWerkstoff - die Werkstoffklasse
 *         string strNennweite - die Nennweite
 *         string strAussendurchmesser - der Aussendurchmesser
 *         string strManteldurchmesser - der Manteldurchmesser
 *		   object oBetriebsbedingungen
 *                double temperatur - Betriebstemperatur
 *                double daemmdicke - Dmmdicke
 *				  double wx - Verschiebung in x-richtung
 *				  double wy - Verschiebung in y-richtung
 *				  double m_fz - Kraft in -z-richtung
 *				  double fz - Kraft in z-richtung
 *				  double m_fx - Kraft in -x-richtung
 *				  double fx - Kraft in x-richtung
 *				  double fy - Kraft in y-richtung
 *         object oZusatzfunktionen - Zusatzfunktionen fr das Lager
 *				  boolean bAbhebesicherung - Abhebesicherung gewhlt?
 *				  boolean bHoehenverstellung - Hhenverstellung gewhlt?
 *				  boolean bGleitunterlage - Gleitunterlage gewhlt?
 *				  boolean bKlemmbar - Klemmbar gewhlt?
 *				  boolean bSchraubbar - schraubbar gewhlt?
 *				  boolean bEinschellig - einschellig gewhlt?
 *				  boolean bBuegelausfuehrung - Bgelausfhrung gewhlt? 
 *         string strAuswahlKontext - Kontext der Lagerauswahl innerhalb der Lastkette
 *
 * @returns	object Suchergebnisse
 *                 boolean bSucheErfolgreich
 *                 string strMeldung - Fehler - oder Warnmeldung
 *                 Array aLager - Array von gefundenen Lagern
 */
Cascade.Lagersuche.sucheLager = function( strLagerart, strGleitart, strGleitelement, strIsolierung, strWerkstoff, strNennweite, strAussendurchmesser, strManteldurchmesser, 
                                          oBetriebsbedingungen, oZusatzfunktionen, strAuswahlKontext )
{
	var oResult =
	{
		bSucheErfolgreich: false,
		strMeldung: '',
		aLager: null
	};
	
	// Vorabprfungen von Temperaturen
	var strError = "";
	
	// Temperatur fr ungedmmte Lager mu zwischen -20C und 80C liegen
	if ( ( strIsolierung == "ID_UNGEDAEMMT" ) && ( ( oBetriebsbedingungen.temperatur < -20 ) || ( oBetriebsbedingungen.temperatur > 80 ) ) )
		strError = translate( "ERROR_RH_MEDIUMTEMP_UNGEDAEMMT" );
	// Temperatur fr wrmegedmmte Lager mu zwischen -20C und 600C liegen
	if ( ( strIsolierung == "ID_WAERMEGEDAEMMT" ) && ( ( oBetriebsbedingungen.temperatur < -20 ) || ( oBetriebsbedingungen.temperatur > 650 ) ) )
		strError = translate( "ERROR_RH_MEDIUMTEMP_GEDAEMMT" );
	// Temperatur fr hhenverstellbare Lager mu <= 30 C sein
	if ( oZusatzfunktionen.bHoehenverstellung && ( oBetriebsbedingungen.temperatur > 300 ) )
		strError = translate( "ERROR_HOEHENV_TEMPER" );
	// Temperatur fr vorisolierte Lager mu <= 450 C sein
	if ( ( strIsolierung == "ID_VORISOLIERT" ) && ( oBetriebsbedingungen.temperatur > 450 ) )
		strError = translate( "ERROR_VORISOLIERT_TEMPER" );
	// Temperatur fr nicht vorisolierte reibungsarme Lager mu <= 300 C sein
	if ( ( strGleitart == "ID_REIBUNGSARM" ) && ( strIsolierung != "ID_VORISOLIERT" ) && ( oBetriebsbedingungen.temperatur > 300 ) )
		strError = translate( "ID_ERRORMESSAGE_REIBUNGSARM_GLEITEND" );

	if ( ( strGleitart == "ID_GLEITBLECH" ) && ( oBetriebsbedingungen.temperatur > 480 ) )
		strError = translate( "ID_ERRORMESSAGE_GLEITBLECH" );

	if ( strError != "" )
	{
		// wenn die Vorabprfung fr Temperaturen fehlschlgt, wird die eigentliche Suche gar nicht mehr durchgefhrt
		oResult.strMeldung = strError;
		return( oResult );
	}
	
	// Vorgabewerte fr Werkstoff
	var bFuehrungVorhanden = Cascade.Lagersuche.isSpeziellesFuehrungslager( strLagerart );
	var werkstoffwerte = Cascade.Lagersuche.berechneVorgabewerte( strLagerart, strIsolierung, strWerkstoff, oBetriebsbedingungen.temperatur, oZusatzfunktionen );

	// Datenbankwerte zur berprfung der Dmmung
	var bCheckdaemmung = false;
	// var nErforderlichesH = 0;  -mw- 05/2008: wir prfen jetzt direkt gegen Variable in den Grunddaten:
	if ( ( strIsolierung != "ID_UNGEDAEMMT" ) && ( strIsolierung != "ID_VORISOLIERT" ) )
	{
		bCheckdaemmung = true;
        /*
        if (strLagerart == "ID_LOSLAGER")
            nErforderlichesH = inputStringToDouble(oBetriebsbedingungen.daemmdicke) + 10;
        else
            nErforderlichesH = inputStringToDouble(oBetriebsbedingungen.daemmdicke) + 30;   // -mw- aufpassen, a) war hier eine Stringaddition, b) knnte Anwender auch Zahl mit Komma eingeben
        */
	}
	
	// Konversion von Eingabewerten fr Zusatzfunktionen in Werte fr die Suche in der Datenbank	
	
	// "Gleitunterlage" bei Los- und Fhrungslager:
	// Werte in DB: 0 (einfach gleitend), 1 (Gleitunterlage am Schlitten), 2 (Gleitunterlage auf Trger)
	//   --> Suche Spalte GL_U bei einfach gleitend nach 0, ansonsten wenn gecheckt nach 2, sonst nach 1
	var nVorgabeWertGleitunterlage = "";
	var bCheckGleitunterlage = false;
	if ( ( ( strLagerart == "ID_LOSLAGER" ) || ( strLagerart == "ID_FUEHRUNGSLAGER" ) ) && ( strGleitart == "ID_REIBUNGSARM" ) )
	{
		bCheckGleitunterlage = true;
		if ( oZusatzfunktionen.bGleitunterlage )
			nVorgabeWertGleitunterlage = "2";
		else nVorgabeWertGleitunterlage = "1";
	}

	// "Bgelausfhrung" bei Festlager ungedmmt
	// Werte in DB: 0 (Bgelausfhrung), 1 (keine Bgelausfhrung)
	//   --> Suche Spalte B_AF wenn gecheckt nach 0, sonst nach 1
	var nVorgabeWertBuegel = ""
	var bCheckBuegelausfuehrung = false;
	if ( ( strLagerart == "ID_FESTLAGER" ) && ( strIsolierung == "IDC_UNGEDAEMMT" ) )
	{
		bCheckBuegelausfuehrung = true;
		if ( oZusatzfunktionen.bBuegelausfuehrung )
			nVorgabeWertBuegel = "0";
		else nVorgabeWertBuegel = "1";
	}

	// "Einschellig" bei DN <= 150 und wrmegedmmt
	// "Einschellig" bei DN <= 300 und Festlager und ungedmmt
	// Werte in DB: 0 (Mehrschellig), 1 (Einschellig)
	//   --> Suche Spalte E_SH wenn gecheckt nach 1, sonst nach 0
	var nVorgabeWertEinschellig = "";
	var bCheckEinschellig = false;
	if ( !oZusatzfunktionen.bSchraubbar && !oZusatzfunktionen.bBuegelausfuehrung )
	{
		if ( ( ( strNennweite <= 150 ) && ( strIsolierung == "ID_WAERMEGEDAEMMT" ) ) ||
			 ( ( strNennweite <= 300 ) && ( strIsolierung == "ID_UNGEDAEMMT" ) && ( strLagerart == "ID_FESTLAGER" ) ) )
		{
			bCheckEinschellig = true;
			if ( oZusatzfunktionen.bEinschellig )
				nVorgabeWertEinschellig = "1";
			else nVorgabeWertEinschellig = "0";
		}
	}
	
	// Fhrung
	var nVorgabeWertFuehrungWx = ""
	if ( bFuehrungVorhanden )
	{
		if ( strLagerart  == "ID_FUEHRUNG_W2_180" )
			nVorgabeWertFuehrungWx = 2;
		else if ( strLagerart  == "ID_FUEHRUNG_W3" )
			nVorgabeWertFuehrungWx = 3;
		else if ( strLagerart  == "ID_FUEHRUNG_W4" )
			nVorgabeWertFuehrungWx = 4;
		else if ( strLagerart  == "ID_FUEHRUNG_W2_180_PLUS" )
			nVorgabeWertFuehrungWx = 5;
		else nVorgabeWertFuehrungWx = 6;
	}
	
	// Manteldurchmesser
	var bCheckManteldurchmesser = ( strIsolierung == "ID_VORISOLIERT" );

	// baue select-statement auf
	var strRelevanteBaureihe = Cascade.Lagersuche.berechneRelevanteBaureihe( strLagerart, strGleitart, strGleitelement, strIsolierung, oZusatzfunktionen, strAuswahlKontext );
	var strDaemmKey = Cascade.Lagersuche.berechneDaemmschluessel( strIsolierung );
	var strSelect = "SELECT * FROM " + strRelevanteBaureihe + " WHERE DN = '" + strNennweite + "' AND DAEM = '" + strDaemmKey + "' order by abs(SEQ_ID) ";
	var aLager = new Array();
	var lager = null;

    // bei der Auswahl der Lager LXL in Lastkette mit Federsttze FSP oder
    // Konstantsttze KSP mssen wir eine Restriktion einbauen.
    // Es muss gelten:  Lagerfubreite A > Durchmesser p  der Tragplatte Sttze
    // Wird bereits als Param bergeben: var strAuswahlKontext   = caMainController.topologieKlasse.getLagerAuswahlKontext();
    var dMassP = 0;
    if ((strAuswahlKontext == "STUETZE_STD") ||(strAuswahlKontext == "STUETZE_PTFE"))
    {
        var Haenger = caMainController.topologieKlasse.getCurrentHaenger();
		var tmpPath = "TMP.XX_ADDITIONAL_DATA";
		if (Haenger.getAdditionalDataToStore(tmpPath) != -1)
            dMassP = getdouble("P", tmpPath);
    }
	
	// Variablen zur Suche
	var strErrorFx = "";
	var strErrorFy = "";
	var strErrorFz = "";
	var strErrorWx = "";
	var strErrorWy = "";
	var nMaxMassWy = 0;
	var nMaxMassWx = 0;
	var strErrorDaemm = "";
	var strErrorFab = "";
	var dWerk_Faktor37 = 0.65;	// wurde bei Werst. 37  fr zul. Belastungen bercksichtigt
	var nMaxLager = 100;		// maximale Anzahl Lager in der Ergebnisliste
	var dH = 0;
    var dJist = 0;
	var nWerkst_RH_DATEN = 0;
	var dWerk_Faktor = 0;
	var bOK = false;
	var strTypBez = "";
	var minFZ = 0;
	var GANZminFZ = 0;
	var minFY = 0;
	var GANZminFY = 0;
	var minFX = 0;
	var GANZminFX = 0;
	var minFAB = 0;
	var GANZminFAB = 0;
	var FX_WERST = 0;
	var FX_KLEMM = 0;

	// suche Daten in Datenbank
	if ( dbselect( "LAGER", strSelect, "RH_DATEN" ) )
	{
		while ( !dbiseof( "RH_DATEN" ) )
		{
			if ( oZusatzfunktionen.bHoehenverstellung )
				dH = dbgetdouble( "RH_DATEN", "H_MAX" ) - 15;
			else dH = dbgetdouble( "RH_DATEN", "H" );

			if ( oZusatzfunktionen.bHoehenverstellung )
				dJist = dbgetdouble( "RH_DATEN", "J_MAX" );
			else dJist = dbgetdouble( "RH_DATEN", "J" );


			bOK = true;
			// teste Werkstoffwerte gegen Werte der Datenbank und berechne Werkstoffaktor
			nWerkst_RH_DATEN = dbgetdouble( "RH_DATEN", "Werk_" );
            var bWerkstoffOK = false;
            for ( var wZaehler = 0; wZaehler < werkstoffwerte.AvailableMaterials.length; ++wZaehler )
                if ( nWerkst_RH_DATEN == werkstoffwerte.AvailableMaterials[wZaehler] )
                    bWerkstoffOK = true;
                    
//			if ( ( nWerkst_RH_DATEN == werkstoffwerte.wert1 ) || ( nWerkst_RH_DATEN == werkstoffwerte.wert2 ) )
            if ( bWerkstoffOK )
			{
				// Abminderungsfaktor entspr. Werkstoff und Temperatur
				if ( nWerkst_RH_DATEN != 37 )
				{
					if ( nWerkst_RH_DATEN == 16 )
					{
						if ( oBetriebsbedingungen.temperatur < 480 )
							dWerk_Faktor = 11.122609 * pow ( oBetriebsbedingungen.temperatur, -0.468394 );
						else dWerk_Faktor = 2.0149 * pow( 10, 29 ) * pow ( oBetriebsbedingungen.temperatur, -11.0101 );
					}
					else if ( nWerkst_RH_DATEN == 13 )
					{
						if ( oBetriebsbedingungen.temperatur < 480 )
							dWerk_Faktor = -0.0003333 * oBetriebsbedingungen.temperatur + 0.91;
						else dWerk_Faktor = 1.57895 * pow( 10, -6 ) * pow( oBetriebsbedingungen.temperatur, 3 ) - 2.404887 * pow( 10, -3 ) * 
						                    pow( oBetriebsbedingungen.temperatur, 2 ) + 1.2111353 * oBetriebsbedingungen.temperatur - 201.125;
					}
					else if ( nWerkst_RH_DATEN == 10 )
					{
                        dWerk_Faktor = 1.9145297 * pow( 10, 21 ) * pow( oBetriebsbedingungen.temperatur, -7.9688 );
                    }
					else if (( nWerkst_RH_DATEN == 41 ) || ( nWerkst_RH_DATEN == 43 ))
					{
                        dWerk_Faktor = 0.000001 * pow( oBetriebsbedingungen.temperatur, 2 ) - 0.0013 * oBetriebsbedingungen.temperatur + 0.9992;
                    }
					else if ( nWerkst_RH_DATEN == 71 )
					{
                        dWerk_Faktor = 0.000001 * pow( oBetriebsbedingungen.temperatur, 2 ) - 0.0015 * oBetriebsbedingungen.temperatur + 1.0644;
                    }
					else if ( nWerkst_RH_DATEN == 80 )
					{
						if ( oBetriebsbedingungen.temperatur <= 600 )
                            dWerk_Faktor = 0.4;
                        else
                            dWerk_Faktor = -9.000001 * pow( oBetriebsbedingungen.temperatur, 2 ) + 0.0099 * oBetriebsbedingungen.temperatur - 2.1233;
                    }
					else if ( nWerkst_RH_DATEN == 91 )
					{
                        dWerk_Faktor = 0.00002075 * pow( oBetriebsbedingungen.temperatur, 2 ) - 0.029887 * oBetriebsbedingungen.temperatur + 10.847;
                    }
					dWerk_Faktor = dWerk_Faktor / dWerk_Faktor37;
				}
				else dWerk_Faktor = 1; // fr Werstoff 37
			}
			else bOK = false;

			// teste Typbezeichnung
			strTypBez = dbgetstring( "RH_DATEN", "TYP" );
			if ( bFuehrungVorhanden )
			{
				if ( ( mid( strTypBez, 5, 1 ) - 0 ) != nVorgabeWertFuehrungWx )
					bOK = false;
			}
			if ( left( strTypBez, 3 ) != 'LUR' )
			{
				if ( ( strTypBez.indexOf( 'XXX' ) > -1 ) || ( ( nWerkst_RH_DATEN == 37 ) && ( strTypBez.length > 16 ) ) )
				{
					// Spezialformen fr Profilarten sind nur in der Datenbank, um die Aufpreise dafr zu kriegen. Sie sollen nicht als Suchergebnisse angezeigt werden
					// Trgerart ist in der Typbezeichnung mit einkodiert -> bis jetzt nur Standardlager (mit Default-Trgerart und -gre) als Suchresultat ausgeben
					bOK = false;
				}
			}
			// else die alten LUR haben eine andere Systematik fr den Typenschlssel
			
			// teste vorhandene Zusatzfunktionen gegen Werte der Datenbank
			if ( bCheckGleitunterlage && ( dbgetstring( "RH_DATEN", "GL_U" ) != nVorgabeWertGleitunterlage ) )
				bOK = false;
			if ( bCheckBuegelausfuehrung && ( dbgetstring( "RH_DATEN", "B_AF" ) != nVorgabeWertBuegel ) )
				bOK = false;
			if ( bCheckEinschellig && ( dbgetstring( "RH_DATEN", "E_SH" ) != nVorgabeWertEinschellig ) )
				bOK = false;
			if ( bCheckManteldurchmesser && ( strManteldurchmesser != ( dbgetstring( "RH_DATEN", "DM" ) ) ) )
				bOK = false;

			// Belastungen und Verschiebungen prfen
			var AchsDirInfo	= caMainController.dataAccess.getAchsKSInfo();
			if ( bOK )
			{
				strErrorDaemm = "";
				strErrorFx 	= "";
				strErrorFy 	= "";
				strErrorFz 	= "";
				strErrorWx 	= "";
				strErrorWy	= "";
				strErrorWy	= "";
				
				// teste Dmmdicke
				// if ( bCheckdaemmung && ( dH < nErforderlichesH ) )
                if (inputStringToDouble(oBetriebsbedingungen.daemmdicke) > dJist)
				{
					bOK = false;
					strErrorDaemm = "#" + translate( "ID_HINWEIS_DAEMMD_ZGROSS" ) + " (J <= " + dJist + " mm)";
				}

				// Belastungen *******************************************************
				
				var VerglOriMinusZ	= "";	// Auflagelast
				var VerglOriPlusZ	= "";	// Abhebelast
				var VerglOriY		= "";	// Axiallast
				var VerglOriX		= "";	// Querlast
				var VerglBewZ		= "";
				var VerglBewY		= "";
				var VerglBewX		= "";
				var PosSenkrRicht	= getstring("SENKR_RICHT", "DLGDATA.CA.FORMS.SYSTEMCHARAKTERISTIK");
				var RohrVerl		= getstring("IDC_ROHRVERLAUF", "DLGDATA.CA.FORMS.SYSTEMCHARAKTERISTIK");
				var RohrAchse		= left(getstring("IDC_ROHRRICHTUNG", "DLGDATA.CA.FORMS.SYSTEMCHARAKTERISTIK"), 1);
				var KraftMatrix		= [oBetriebsbedingungen.fx, oBetriebsbedingungen.fy,
										oBetriebsbedingungen.fz, oBetriebsbedingungen.m_fx,
										oBetriebsbedingungen.m_fy, oBetriebsbedingungen.m_fz];
				var BewegungMatrix	= [oBetriebsbedingungen.wx, oBetriebsbedingungen.wy,
										oBetriebsbedingungen.wz, oBetriebsbedingungen.m_wx,
										oBetriebsbedingungen.m_wy, oBetriebsbedingungen.m_wz];
				
				// -Schu- Zuordnung der Kraefte und Bewegungen auf das "alte" feste KS
				if (RohrVerl == "HORIZONTAL")
				{
					var IndexAuflDirMi	= AchsDirInfo[6] + 3;
					var IndexAuflDirPl	= AchsDirInfo[6];
					
					if (RohrAchse == AchsDirInfo[1])
					{
						var IndexAxDir		= AchsDirInfo[7];
						var IndexQuerDir	= AchsDirInfo[8];
						
					}
					else
					{
						var IndexAxDir		= AchsDirInfo[8];
						var IndexQuerDir	= AchsDirInfo[7];
					}
					
					if (PosSenkrRicht != "AUF")
					{
						IndexAuflDirMi	-=	3;
						IndexAuflDirPl	+=	3;
					}
					
					VerglOriMinusZ	= KraftMatrix[IndexAuflDirMi];
					VerglOriPlusZ	= KraftMatrix[IndexAuflDirPl];
					VerglOriY		= max(abs(KraftMatrix[IndexAxDir]), abs(KraftMatrix[IndexAxDir + 3]));
					VerglOriX		= max(abs(KraftMatrix[IndexQuerDir]), abs(KraftMatrix[IndexQuerDir + 3]));
					VerglBewZ		= max(abs(BewegungMatrix[IndexAuflDirMi]), abs(BewegungMatrix[IndexAuflDirPl + 3]));
					VerglBewY		= max(abs(BewegungMatrix[IndexAxDir]), abs(BewegungMatrix[IndexAxDir + 3]));
					VerglBewX		= max(abs(BewegungMatrix[IndexQuerDir]), abs(BewegungMatrix[IndexQuerDir + 3]));
				}				
				else
				{
					if (right(getstring("IDC_STUETZRICHTUNG_IST", "DLGDATA.CA.FORMS.LAGER_BETRIEBSBEDINGUNGEN"), 1) == AchsDirInfo[1])
					{
						var IndexQuerDir	= AchsDirInfo[8];
						var IndexAuflDirMi	= AchsDirInfo[7];
						var IndexAuflDirPl	= AchsDirInfo[7] + 3;
					}
					else
					{
						var IndexQuerDir	= AchsDirInfo[7];
						var IndexAuflDirMi	= AchsDirInfo[8];
						var IndexAuflDirPl	= AchsDirInfo[8] + 3;
					}
					
					if (left(getstring("IDC_STUETZRICHTUNG_IST", "DLGDATA.CA.FORMS.LAGER_BETRIEBSBEDINGUNGEN"), 2) != "PL")
					{
						IndexAuflDirPl	+=	3;
						IndexAuflDirMi	-=	3;
					}
					
					VerglOriMinusZ	= KraftMatrix[IndexAuflDirMi];
					VerglOriPlusZ	= KraftMatrix[IndexAuflDirPl];
					VerglBewZ		= max(abs(BewegungMatrix[IndexAuflDirMi]), abs(BewegungMatrix[IndexAuflDirPl]));
					VerglOriY		= max(abs(KraftMatrix[AchsDirInfo[6]]), abs(KraftMatrix[AchsDirInfo[6] + 3]));
					VerglBewY		= max(abs(BewegungMatrix[AchsDirInfo[6]]), abs(BewegungMatrix[AchsDirInfo[6] + 3]));
					VerglOriX		= max(abs(KraftMatrix[IndexQuerDir]), abs(KraftMatrix[IndexQuerDir + 3]));
					VerglBewX		= max(abs(BewegungMatrix[IndexQuerDir]), abs(BewegungMatrix[IndexQuerDir + 3]));
				}
				
				// teste -fz
				minFZ = dbgetdouble( "RH_DATEN", "FZ" ) * dWerk_Faktor;
				//if ( minFZ < oBetriebsbedingungen.m_fz )
				if ( minFZ < VerglOriMinusZ )
				{
					bOK = false;
					if ( find( "" + minFZ, "." ) == -1 )
					   GANZminFZ = minFZ;
					else GANZminFZ = mid( "" + minFZ, 1, find( "" + minFZ, "." ) - 1 );
					strErrorFz = "#" + translate( "ERRORMESSAGE_FZ_" ) + "  ( -Fz <= " + GANZminFZ + " kN )";
				}

				// teste Belastungen fx und fy 
				if ( ( strLagerart == "ID_FESTLAGER" ) || ( strLagerart == "ID_FESTLAGER_D" ) )
				{
					// teste fy
					if ( oZusatzfunktionen.bKlemmbar )
						minFY = dbgetdouble( "RH_DATEN", "FY" );
					else minFY = dbgetdouble( "RH_DATEN", "FY" ) * dWerk_Faktor;
					//if ( minFY < oBetriebsbedingungen.fy )
					if ( minFY < VerglOriY )
					{
						bOK = false;
						if ( find( "" + minFY, "." ) == -1 )
							GANZminFY = minFY;
						else GANZminFY = mid( "" + minFY, 1, find( "" + minFY, "." ) -1 );
						strErrorFy = "#" + translate( "ERRORMESSAGE_FY" ) + "  ( Fy <= " + GANZminFY + " kN )";
					}
					
					// teste fx
					if ( oZusatzfunktionen.bKlemmbar )
					{
						FX_WERST = dbgetdouble( "RH_DATEN", "FX" ) * dWerk_Faktor;
						FX_KLEMM = dbgetdouble( "RH_DATEN", "FX_K" );
						if ( FX_WERST < FX_KLEMM )
							minFX = FX_WERST;
						else minFX = FX_KLEMM;
					}
					else minFX = dbgetdouble( "RH_DATEN", "FX" ) * dWerk_Faktor;
					
					//if ( minFX < max(abs(oBetriebsbedingungen.fx), abs(oBetriebsbedingungen.m_fx)) )
					if ( minFX < VerglOriX )
					{
						bOK = false;
						if ( find( "" + minFX, "." ) == -1 )
							GANZminFX = minFX;
						else GANZminFX = mid( "" + minFX, 1, find( "" + minFX, "." ) -1 );
						strErrorFx = "#" + translate( "ERRORMESSAGE_FX" ) + "  ( Fx <= " + GANZminFX + " kN )";
					}
				}
				else
				{
				// teste Belastungen fx und fy fr Fhrungslager
				if ( ( strLagerart == "ID_FUEHRUNGSLAGER" ) || 
                         ( strLagerart == "ID_FUEHRUNGSLAGER_W" ) ||
                         ( bFuehrungVorhanden && ( nVorgabeWertFuehrungWx != 2 ) ) )
					{
						if ( oZusatzfunktionen.bKlemmbar )
						{
							FX_WERST = dbgetdouble( "RH_DATEN", "FX" ) * dWerk_Faktor;
							FX_KLEMM = dbgetdouble( "RH_DATEN", "FX_K" );
							if ( FX_WERST < FX_KLEMM )
								minFX = FX_WERST;
							else minFX = FX_KLEMM;
						}
						// Ergnzung scb
						else if (strLagerart == "ID_FUEHRUNGSLAGER_W" )
						{
							FX_WERST = dbgetdouble( "RH_DATEN", "FX" ) * dWerk_Faktor;
							FX_KLEMM = dbgetdouble( "RH_DATEN", "FX_W" );
							if ( FX_WERST < FX_KLEMM )
								minFX = FX_WERST;
							else minFX = FX_KLEMM;
						}
						// ----------------------------------------------------------------

						else minFX = dbgetdouble( "RH_DATEN", "FX" ) * dWerk_Faktor;

						//if  ( minFX < max(abs(oBetriebsbedingungen.fx), abs(oBetriebsbedingungen.m_fx)) )
						if  (minFX < VerglOriX)
						{
							bOK = false;
							if ( find( "" + minFX, "." ) == -1 )
								GANZminFX = minFX;
							else GANZminFX = mid( "" + minFX, 1, find( "" + minFX, "." ) -1 );
							strErrorFx = "#" + translate( "ERRORMESSAGE_FX" ) + "  ( Fx <= " + GANZminFX + " kN )"
						}
					}
					// fr Loslager brauchen Belastungen fx und fy nicht getestet zu werden
				}
				
				// teste Belastung fz
				if ( ( strLagerart != "ID_LOSLAGER" ) || ( nVorgabeWertFuehrungWx != 6 ) )
				{
					if (( oBetriebsbedingungen.temperatur > 300 ) && (strLagerart != "ID_FUEHRUNGSLAGER_W" ))  // Ergnzung scb
					{
						FAB_WERST = dbgetdouble( "RH_DATEN", "FAB" ) * dWerk_Faktor;
						FAB_KLEMM = dbgetdouble( "RH_DATEN", "FAB_K" );
						if ( FAB_KLEMM == 0 )
							minFAB = FAB_WERST;
						else
						{
							if ( FAB_WERST < FAB_KLEMM )
								minFAB = FAB_WERST;
							else minFAB = FAB_KLEMM;
						}
					}
					// Ergnzung scb
					else if (strLagerart == "ID_FUEHRUNGSLAGER_W" )
					{
						FAB_WERST = dbgetdouble( "RH_DATEN", "FAB" ) * dWerk_Faktor;
						FAB_KLEMM = dbgetdouble( "RH_DATEN", "FAB_W" );
						if ( FAB_WERST < FAB_KLEMM )
							minFAB = FAB_WERST;
						else minFAB = FAB_KLEMM;
					}
					// ----------------------------------------------------------------
					else minFAB = dbgetdouble( "RH_DATEN", "FAB" );

					//if  ( minFAB < oBetriebsbedingungen.fz )
					if  ( minFAB < VerglOriPlusZ )
					{
						bOK = false;
						if ( find( "" + minFAB, "." ) == -1 )
							GANZminFAB = minFAB;
						else GANZminFAB = mid( "" + minFAB, 1, find( "" + minFAB,"." ) -1 );
						strErrorFab = "#" + translate( "ERRORMESSAGE_FAB" ) + "  ( Fab <= " + GANZminFAB + " kN )";
					}
				}

				// Runden der korrigierten Belastungen
				minFX = Cascade.Lagersuche.rundeBelastung( minFX );
				minFY = Cascade.Lagersuche.rundeBelastung( minFY );
				minFZ = Cascade.Lagersuche.rundeBelastung( minFZ );
				minFAB = Cascade.Lagersuche.rundeBelastung( minFAB );

				// teste Verschiebungen    *******************************************************
                
				// teste wx fr Loslager auf Sttze
                if ((strAuswahlKontext == "STUETZE_STD") || (strAuswahlKontext == "STUETZE_PTFE"))
                {
                    var dFaktor = 0;
					var dA = dbgetdouble( "RH_DATEN", "A" ); 
                    if (left(strRelevanteBaureihe, 3) == "LXL")
					{
                        dFaktor = 10;
						var zulWX = (dA - dFaktor) - dMassP;
					}
					else
					{
						var zulWX = 0.2 * dA
					}
					//if (zulWX < oBetriebsbedingungen.wx)
					if (zulWX < VerglBewX)
					{
						bOK = false;
						if (nMaxMassWx < zulWX)
							nMaxMassWx = zulWX;
						// strErrorWx = "#" + translate( "ERRORMESSAGE_WX" ) +  " " + dbgetdouble( "RH_DATEN", "WX" ) + " mm ) !";
						strErrorWx = "#" + translate( "ERRORMESSAGE_WX" ) +  " " + nMaxMassWx + " mm ) !";
					}
                }
				// Check zustzlich Lagerfubreite fr LXL, s.o.
				if ( (dMassP > 0) && (left(strRelevanteBaureihe, 3) == "LXL") )
				{
					var dLagerfussBreite = dbgetdouble( "RH_DATEN", "A" );
					if (dLagerfussBreite < dMassP)
						bOK = false;
				}
				// teste wx fr Loslager mit Gleitunterlage
				if ( ( strLagerart == "ID_LOSLAGER" ) && oZusatzfunktionen.bGleitunterlage )
				{
					//if ( ( dbgetdouble( "RH_DATEN", "WX" ) > 0.1 ) && ( dbgetdouble( "RH_DATEN", "WX" ) < oBetriebsbedingungen.wx ) )
					if ( ( dbgetdouble( "RH_DATEN", "WX" ) > 0.1 ) && ( dbgetdouble( "RH_DATEN", "WX" ) < VerglBewX ) )
					{
						bOK = false;
						strErrorWx = "#" + translate( "ERRORMESSAGE_WX" ) +  " " + dbgetdouble( "RH_DATEN", "WX" ) + " mm ) !";
					}
				}
				// teste wx fr Loslager mit Gleitblech und Gleitelement
				if ( ( strLagerart == "ID_LOSLAGER" ) && ( strGleitart == "ID_GLEITBLECH" ) && 
						(( strGleitelement == 'ID_LGA' ) || ( strGleitelement == 'ID_LGV' )) )
				{
					//if ( ( dbgetdouble( "RH_DATEN", "WX" ) > 0.1 ) && ( dbgetdouble( "RH_DATEN", "WX" ) < oBetriebsbedingungen.wx ) )
					if ( ( dbgetdouble( "RH_DATEN", "WX" ) > 0.1 ) && ( dbgetdouble( "RH_DATEN", "WX" ) < VerglBewX ) )
					{
						bOK = false;
						strErrorWx = "#" + translate( "ERRORMESSAGE_WX" ) +  " " + dbgetdouble( "RH_DATEN", "WX" ) + " mm ) !";
					}
				}
				
				// teste wy fr Loslager, Loslager mit Gleitblech und Gleitelement, Fhrungslager und Fhrungen
				if ( ( ( strLagerart == "ID_LOSLAGER" ) || ( strLagerart == "ID_FUEHRUNGSLAGER" ) || 
						( strLagerart == "ID_FUEHRUNGSLAGER_W" ) || bFuehrungVorhanden ) &&
						( ( strIsolierung != "ID_UNGEDAEMMT" ) || 
						( ( strIsolierung == "ID_UNGEDAEMMT" ) && !oZusatzfunktionen.bGleitunterlage ) ) )
				{
					if ( ( strLagerart == "ID_LOSLAGER" ) && ( strGleitart == "ID_GLEITBLECH" ) && ( strGleitelement == 'ID_OHNE' ) )
					// Diese Lager werden blicherweise mit Gleitelement verwendet.
					// Verschiebung WY muss deshalb um anteilige Lnge (50 mm sind schon bercksichtigt) des Gleitelementes verringert werden.
					{
						// Nherungsweise Berechnung der Lnge des Gleitelementes
						LAENGE_LGX = 0;
						if ( ( mid( strTypBez, 5, 2 ) - 0 ) <= 22 )
							LAENGE_LGX = 20;
						else if ( strNennweite < 200 )
							LAENGE_LGX = 20;
						else
							LAENGE_LGX = 70;
						
						//if ( ( dbgetdouble( "RH_DATEN", "WY" ) > 0.1 ) && ( (dbgetdouble( "RH_DATEN", "WY" ) - LAENGE_LGX) < oBetriebsbedingungen.wy ) )
						if ( ( dbgetdouble( "RH_DATEN", "WY" ) > 0.1 ) && ( (dbgetdouble( "RH_DATEN", "WY" ) - LAENGE_LGX) < VerglBewY ) )
						{
							bOK = false;
							strErrorWy = "#" + translate( "ERRORMESSAGE_WY" ); // + " " + (dbgetdouble("RH_DATEN", "WY") - LAENGE_LGX) + " mm ) !";
							if (nMaxMassWy < (dbgetdouble("RH_DATEN", "WY") - LAENGE_LGX))
								nMaxMassWy = (dbgetdouble("RH_DATEN", "WY") - LAENGE_LGX);
						}
					}
					//else if ( ( dbgetdouble( "RH_DATEN", "WY" ) > 0.1 ) && ( dbgetdouble( "RH_DATEN", "WY" ) < oBetriebsbedingungen.wy ) )
					else if ( ( dbgetdouble( "RH_DATEN", "WY" ) > 0.1 ) && ( dbgetdouble( "RH_DATEN", "WY" ) < VerglBewY ) )
					{
						bOK = false;
						strErrorWy = "#" + translate( "ERRORMESSAGE_WY" );  // + " " + dbgetdouble("RH_DATEN", "WY") + " mm ) !";
						if (nMaxMassWy < dbgetdouble("RH_DATEN", "WY"))
							nMaxMassWy = dbgetdouble("RH_DATEN", "WY");
					}
				}
			}
            
			// der Kandidat hat alle technischen Prfungen berstanden -> erzeuge Lager und fge in Lagerliste ein
			// Achtung: es werden maximal nMaxLager Lager in die Ergebnisliste eingefgt
			if ( bOK && ( aLager.length < nMaxLager ) )
			{
				// erzeuge neues Lager kopiere alle Datenbankwerte in das neue Lagerobjekt
				lager = new Lager( "RH_DATEN", oBetriebsbedingungen );
				//dbcopy( "RH_DATEN", ResultListeI )

                // LSW soll nicht gezeigt werden, da aus 2 Typen zusammengesetzt
                //if (lager.strTyp.left(3) == "LSW"))
                //    lager.strTyp = "..." + lager.strTyp.right(lager.strTyp.length - 3);
                    
				// korrigiere Werte fr Oberflchenanzeige
				if ( ( strLagerart == "ID_FESTLAGER" ) || ( strLagerart == "ID_FESTLAGER_D" ) )
				{
					lager.strNennlast = "" + minFY;
					lager.strNennlastName = "Fy";
				}
				else
				{
					lager.strNennlast = "" + minFZ;
					lager.strNennlastName = "-Fz";
				}

				if ( strLagerart != "ID_LOSLAGER" )
					lager.strFX = "" + minFX;

				if ( oZusatzfunktionen.bHoehenverstellung )
				{
					lager.strHoeheMinMax = dbgetstring( "RH_DATEN", "H_MIN_MAX" );
					lager.strHoeheMin = dbgetstring( "RH_DATEN", "H_MIN" );
					lager.strHoeheMax = dbgetstring( "RH_DATEN", "H_MAX" );
				}
				else lager.strHoeheMinMax = "";

				// if ( ( strLagerart == "ID_LOSLAGER" ) || ( bFuehrungVorhanden || ( ( nVorgabeWertFuehrungWx == 3 ) || ( nVorgabeWertFuehrungWx == 6 ) ) ) )
				if ( ( strLagerart == "ID_LOSLAGER" ) || ( nVorgabeWertFuehrungWx == 3 ) || ( nVorgabeWertFuehrungWx == 6 ) ) 
					lager.strFAB = "-";
				else lager.strFAB = "" + minFAB;
				
				if ( nVorgabeWertFuehrungWx == 2 )
					lager.strFX = "-";
					
				if ( ( strLagerart != "ID_FESTLAGER" ) && ( strLagerart != "ID_FESTLAGER_D" ) )
					lager.strFY = "-";
                else // -mw- 09/2008:
                    lager.strFY = "" + minFY;
                
                // -mw- 09/2008:
                lager.strFZ = "" + minFZ;
				
				lager.strBasistyp = dbgetstring( "RH_DATEN", "TYP" );
				lager.strBaureihe = strRelevanteBaureihe;

				// in Lagerliste einfgen
				aLager.push( lager );
			}

			// nchster Record
			dbskip( "RH_DATEN" );
		}
	}

	// Rckgabewerte zusammenstellen
	if ( aLager.length <= 0 ) // nichts gefunden
	{
		strError = translate( "ERROR_KEIN_PRODUKT_GEFUNDEN" ) + " ";
		strError = strError + strErrorFx + strErrorFy + strErrorFz + strErrorFab + strErrorWx + strErrorDaemm;
		if (strErrorWy != "")
			strError += strErrorWy + " " + nMaxMassWy + " mm ) !"
		oResult.strMeldung = strError;
	}
	else
	{
		oResult.bSucheErfolgreich = true;
		oResult.aLager = aLager;
	}

	return( oResult );
}

/**
 * speichere die bergebene Liste
 *
 * @params array aListe - Liste mit Lagerobjekten
 * @returns	-
 */
Cascade.Lagersuche.speichereLagerliste = function( aListe )
{
	// lsche letzte Ergebnisliste
	var strBaseStorePath = Cascade.Lagersuche._strTempStorePath;
	delete( strBaseStorePath, "" );
	
	if ( !aListe )
	{
		// keine Ergebnisliste zwischenzuspeichern
		setstring( "COUNT", strBaseStorePath, "0" );
		return;
	}

	// anzahl der Lager in der Liste
	var nCount = aListe.length;
	setstring( "COUNT", strBaseStorePath, "" + nCount );
	
	// speichere alle Lager
	var strListPath = strBaseStorePath + ".RESULT_LIST";
	var strItemPath;
	for ( var i = 0; i < nCount; ++i )
	{
		// speichere jedes Lager im Store
		strItemPath = strListPath + "[" + i + "]";
		aListe[i].serializeOut( strItemPath );
	}	
}


/**
 * rekonstruiere die aktuell im Store zwischengespeicherte Lagerliste, das Suchresultat der letzten erfolgreich durchgefhrten Suche,
 * als Javascript-Objektstruktur
 *
 * @params -
 * @returns	array - Array mit Lagerobjekten
 */
Cascade.Lagersuche.rekonstruiereLetzeLagerliste = function()
{
	var aResult = new Array();
	var strBaseStorePath = Cascade.Lagersuche._strTempStorePath;

	// teste, ob berhaupt schon eine Suchliste zwischengespeichert wurde
	if ( !checkexist( strBaseStorePath, "" ) )
		return( aResult );
		
	var nCount = getstring( "COUNT", strBaseStorePath ) - 0;
	if ( nCount <= 0 )	// Suchliste der letzten Suche war leer
		return( aResult );
	
	// rekonstruiere jeder Lager der Liste
	var strListPath = strBaseStorePath + ".RESULT_LIST";
	var strItemPath;
	var lager;
	for( var i = 0; i < nCount; ++i )
	{
		strItemPath = strListPath + "[" + i + "]";
		
		// lade Lagerdaten aus Store
		lager = new Lager();
		lager.serializeIn( strItemPath );
		aResult.push( lager );
	}
	
	return( aResult );
}


/**
 * hole das Lagerobjekt mit dem angegebenen Index aus der aktuell im Store zwischengespeicherten Lagerliste
 *
 * @params integer nIndex - Index(0-basiert) des gewhlten Lagers in der Lagerliste
 * @returns	object - aktuell gewhltes Lager
 */
Cascade.Lagersuche.holeLagerAusLagerliste = function( nIndex )
{
	var strBaseStorePath = Cascade.Lagersuche._strTempStorePath;

	// teste, ob berhaupt schon eine Suchliste zwischengespeichert wurde
	if ( !checkexist( strBaseStorePath, "" ) )
		return( null );
		
	var nCount = getstring( "COUNT", strBaseStorePath ) - 0;
	if ( nIndex >= nCount )	// Suchliste der letzten Suche war leer, oder Index passt nicht mehr zur aktuellen Suchliste
		return( null );
	
	// rekonstruiere Lager mit angegebenem Index
	var strItemPath = strBaseStorePath + ".RESULT_LIST[" + nIndex + "]";
	var lager = new Lager()
	lager.serializeIn( strItemPath );
	
	return( lager );
}


