/**
 * CascadeKlemmsystem.es: modul mit Utilityfunktionen fr die Suche von Klemmsystemen
 * Alle Utilityfunktionen liegen in einem Namespace 'Cascade.Klemmsystem'
 *
 **/

// Namespace erzeugen
if ( typeof( Cascade ) == "undefined" )
	Cascade = {};	
if ( !Cascade.Klemmsystem )
	Cascade.Klemmsystem = {};
	
/**
 * @var Storepfad fr temporres Zwischenspeichern von gefundenen Klemmsystemlisten
 */
 
Cascade.Klemmsystem._strTempStorePath = "DLGDATA.TEMP_RESULTS.KLEMMSYSTEM_AUSWAHL";

/**
 * berechnet die fr die Suche relevante Baureihe des Klemmsystems
 *
 * @params string strLagerart - die Lagerart
 *         string strTraegerart - die Trgerprofilart
 *         object oZusatzfunktionen - Zusatzfunktionen fr das Lager
 *				  boolean bAbhebesicherung - Abhebesicherung gewhlt?
 *				  boolean bAbhebesicherung4fach - Abhebesicherung 4-fach gewhlt?
 *				  boolean bGleitunterlage - Gleitunterlage gewhlt?
 *                string strBreite - Lagerfubreite
 * @returns	string Kennung der relevanten Baureihe
 */
Cascade.Klemmsystem.berechneRelevanteBaureihe = function( strLagerart, strTraegerart, oZusatzfunktionen )
{
	var strFirstChar = "K";
	var strSecondChar = "";
	var strThirdChar = "";
	
	// zweites Zeichen
	if ( strLagerart == "ID_FUEHRUNGSLAGER" )
	{
		if ( oZusatzfunktionen.bAbhebesicherung )
		{
			if ( oZusatzfunktionen.bAbhebesicherung4fach )
				strSecondChar = "A"
			else strSecondChar = "Z"
		}
		else strSecondChar = "O"
	}
	else strSecondChar = "L"

	// drittes Zeichen
	if ( strTraegerart == "U_PROFIL" )
		strThirdChar = "U"
	else if ( ( strTraegerart == "L_PROFIL_G" ) || ( strTraegerart == "L_PROFIL_U" ) )
		strThirdChar = "L"
	else strThirdChar = "T"

	return( strFirstChar + strSecondChar + strThirdChar );
}


/**
 * sucht zu den Eingabewerten passende Klemmsysteme
 *
 * @params string strLagerart - die Lagerart
 *         string strTraegerart - die Trgerprofilart
 *         string strTraegergroesse - die Trgergre
 *         object oZusatzfunktionen - Zusatzfunktionen fr das Klemmsystem
 *				  boolean bAbhebesicherung - Abhebesicherung gewhlt?
 *				  boolean bAbhebesicherung4fach - Abhebesicherung 4-fach gewhlt?
 *				  boolean bGleitunterlage - Gleitunterlage gewhlt?
 *                string strBreite - Lagerfubreite
 * @returns	object Suchergebnisse
 *                 boolean bSucheErfolgreich
 *                 string strMeldung - Fehler - oder Warnmeldung
 *                 Array aKlemmsysteme - Array von gefundenen Klemmsystemen
 */
Cascade.Klemmsystem.sucheKlemmsystem = function( strLagerart, strTraegerart, strTraegergroesse, oZusatzfunktionen )
{
	var oResult =
	{
		bSucheErfolgreich: false,
		strMeldung: '',
		aLager: null
	};
	
	var strError = "";

	// Typenschlssel
	var strVorgabeWertTyp = Cascade.Klemmsystem.berechneRelevanteBaureihe( strLagerart, strTraegerart, oZusatzfunktionen );
	
	// Suffix fr den Typenschlssel bestimmen
	var strTypenschluesselSuffix = "";
	var oTraegerdaten = Cascade.Lagerkonfiguration.holeTraegerdaten( strTraegerart, strTraegergroesse, null );
	if ( ( ( strTraegerart == "HEB" ) || ( strTraegerart == "HEA" ) || ( strTraegerart == "IPE" ) || ( strTraegerart == "INP" ) || ( strTraegerart == "T_PROFIL" ) ) &&
		 ( oTraegerdaten.strVarKlemm == "T140" ) )
		strTypenschluesselSuffix = "" ; 								// Standard, Variante Werkstoff/Oberflche ist hier nicht interessant
	else strTypenschluesselSuffix = "-" + oTraegerdaten.strVarKlemm;	// Spezial, Variante Werkstoff/Oberflche steht im Typenschlssel
	
	// Breite Lagerfuss bei Gleitunterlage am Klemmsystem
    var bCheckGleitunterlage = oZusatzfunktionen.bGleitunterlage;
	var strVorgabeWertGleitunterlage = "0";
	var strVorgabeFussBreite = "";
	if ( oZusatzfunktionen.bGleitunterlage )
	{
		strVorgabeFussBreite = mid( oZusatzfunktionen.strBreite, 5, 7 );
	   	strVorgabeWertGleitunterlage = "2";
    }
	
    // Trgerbreitenbereich bei T-Trgern
    var bCheckKlemmbreite = false;
	var strVorgabeKlemm = "0";
	if ( ( ( strTraegerart == "HEB" ) || ( strTraegerart == "HEA" ) || ( strTraegerart == "IPE" ) || ( strTraegerart == "INP" ) || ( strTraegerart == "T_PROFIL" ) ) &&
	     oZusatzfunktionen.bGleitunterlage )
	{
		bCheckKlemmbreite = true;
		if ( oTraegerdaten.strVarKlemm == "T140" )
			strVorgabeKlemm = "140"
		else strVorgabeKlemm = "300"
	}

	// Datenbankvorgaben fr Profilform und Trgerbreite
	var strVorgabeTraegerbreite	= "";
	var strVorgabeProfilform = "T";
	if ( ( strTraegerart == "HEB" ) || ( strTraegerart == "HEA" ) || ( strTraegerart == "IPE" ) || ( strTraegerart == "INP" ) || ( strTraegerart == "T_PROFIL" ) )
	{
		//strVorgabeProfilform = "T";		// das ist sowieso der Grundwert
		if ( oTraegerdaten.strVarKlemm == "T140" )
			strVorgabeTraegerbreite = "140";
	}
	else if ( strTraegerart == "U_PROFIL" )
		strVorgabeProfilform = "U";
	else if ( ( strTraegerart == "L_PROFIL_G" ) || ( strTraegerart == "L_PROFIL_U" ) )
		strVorgabeProfilform = "L";
	
	// baue select-statement
	var strSelect = "SELECT * FROM KXX WHERE SUBSTR( TYP, 1, 3 ) = '";
	strSelect += strVorgabeWertTyp;
	strSelect += "' AND GL_U = '";
	strSelect += strVorgabeWertGleitunterlage;
	strSelect += "' AND PROFILFORM = '";
	strSelect += strVorgabeProfilform;
	if ( strVorgabeProfilform == 'T' )
	{
		strSelect += "' AND TRAEGERBREITE = '";
		strSelect += strVorgabeTraegerbreite;
	}
	strSelect += "' ";
	
	// Suche nach Klemmsystemen starten
	var aKlemmsysteme = new Array();
	var ks = null;
	var bOK = false;
	var dbTyp = "";
	var dbTyp3 = "";
	var dbTypFussBreite = "";
	
	if ( dbselect( "LAGER", strSelect, "RH_DATEN" ) )
	{
		while ( !dbiseof( "RH_DATEN" ) )
		{
			// teste Typenschlssel
			dbTyp = dbgetstring( "RH_DATEN", "TYP" );

			bOK = true;
			if ( bCheckGleitunterlage && ( strlen( dbTyp ) > 9 ) )
			{
				// teste Lagerfubreite und Klemmsystemvarianten, wenn fr Gleitunterlage ntig
				bOK = false;
				dbTypFussBreite = mid( dbTyp, 8, 3 );
				if ( strVorgabeFussBreite == dbTypFussBreite )
				{
					bOK = true;
					// Test auf maximale Klemmsystembreite entfallen, da keine Daten mehr in der Datenbank
					// if ( bCheckKlemmbreite )
					//   	bOK = ( dbgetstring( "RH_DATEN", "K" ) == strVorgabeKlemm );
				}
			}

			// der Kandidat hat alle technischen Prfungen berstanden -> erzeuge Klemmsystem und fge in Klemmsystemliste ein
			if ( bOK )
			{
				// erzeuge neues Lager kopiere alle Datenbankwerte in das neue Lagerobjekt
				ks = new Klemmsystem( "RH_DATEN" );

				// korrigiere Daten des Klemmsystems
				ks.strBasistyp = dbTyp;
				ks.strTyp = Cascade.Klemmsystem.korrigiereTypenschluessel( dbTyp, strTypenschluesselSuffix );
				ks.strBaureihe = "KXX";
				ks.strProfilart = strTraegerart;
				ks.strTraegergroesse = strTraegergroesse;
				ks.strTraegerhoehe = oTraegerdaten.strTraegerhoehe;
				ks.strTraegerbreite = oTraegerdaten.strTraegerbreite;
				ks.strFlanschdicke = oTraegerdaten.strFlanschdicke;

				// in Lagerliste einfgen
				aKlemmsysteme.push( ks );
			}

			// nchster Record
			dbskip( "RH_DATEN" );
		}
	}

	// Rckgabewerte zusammenstellen
	if ( aKlemmsysteme.length <= 0 ) // nichts gefunden
		oResult.strMeldung = translate( "ERROR_KEIN_PRODUKT_GEFUNDEN" );
	else
	{
		oResult.bSucheErfolgreich = true;
		oResult.aKlemmsysteme = aKlemmsysteme;
	}

	return( oResult );
}

/**
 * korrigiert den Typenschlssel eines Klemmsystems
 *
 * @params string strDBTyp - ursprnglicher Typenschlssel aus der Datenbank
 *         string strSuffix - neues Suffix fr korrekte Trgerbreite
 * @returns	string - der korrigierte Typenschlssel
 */
Cascade.Klemmsystem.korrigiereTypenschluessel = function( strDBTyp, strSuffix )
{
	var strKorrigierterTyp = strDBTyp;
	var nIndex = strDBTyp.indexOf( '-TXXX' );	// spezielle T-Profile
	if ( nIndex == -1 )
		nIndex = strDBTyp.indexOf( '-LXXX' );	// spezielle L-Profile
	if ( nIndex == -1 )
		nIndex = strDBTyp.indexOf( '-UXXX' );	// spezielle U-Profile
	// else Profil mit Standardtraegerbreite
	
	if ( nIndex > -1 )
		strKorrigierterTyp = left( strDBTyp, nIndex ) + strSuffix;
		
	return( strKorrigierterTyp );
}


/**
 * speichere die bergebene Liste
 *
 * @params array aListe - Liste mit Klemmsystemobjekten
 * @returns	-
 */
Cascade.Klemmsystem.speichereKlemmsystemliste = function( aListe )
{
	// lsche letzte Ergebnisliste
	var strBaseStorePath = Cascade.Klemmsystem._strTempStorePath;
	delete( strBaseStorePath, "" );
	
	if ( !aListe )
	{
		// keine Ergebnisliste zwischenzuspeichern
		setstring( "COUNT", strBaseStorePath, "0" );
		return;
	}

	// anzahl der Klemsysteme in der Liste
	var nCount = aListe.length;
	setstring( "COUNT", strBaseStorePath, "" + nCount );
	
	// speichere alle Klemmsysteme
	var strListPath = strBaseStorePath + ".RESULT_LIST";
	var strItemPath;
	for ( var i = 0; i < nCount; ++i )
	{
		// speichere jedes Klemmsystem im Store
		strItemPath = strListPath + "[" + i + "]";
		aListe[i].serializeOut( strItemPath );
	}	
}


/**
 * rekonstruiere die aktuell im Store zwischengespeicherte Klemmsystemliste, das Suchresultat der letzten erfolgreich durchgefhrten Suche,
 * als Javascript-Objektstruktur
 *
 * @params -
 * @returns	array - Array mit Klemmsystemobjekten
 */
Cascade.Klemmsystem.rekonstruiereLetzeKlemmsystemliste = function()
{
	var aResult = new Array();
	var strBaseStorePath = Cascade.Klemmsystem._strTempStorePath;

	// teste, ob berhaupt schon eine Suchliste zwischengespeichert wurde
	if ( !checkexist( strBaseStorePath, "" ) )
		return( aResult );
		
	var nCount = getstring( "COUNT", strBaseStorePath ) - 0;
	if ( nCount <= 0 )	// Suchliste der letzten Suche war leer
		return( aResult );
	
	// rekonstruiere jedes Klemmsystem der Liste
	var strListPath = strBaseStorePath + ".RESULT_LIST";
	var strItemPath;
	var ks;
	for( var i = 0; i < nCount; ++i )
	{
		strItemPath = strListPath + "[" + i + "]";
		
		// lade Klemmsystemdaten aus Store
		ks = new Klemmsystem();
		ks.serializeIn( strItemPath );
		aResult.push( ks );
	}
	
	return( aResult );
}


/**
 * hole das Klemmsystemobjekt mit dem angegebenen Index aus der aktuell im Store zwischengespeicherten Klemmsystemliste
 *
 * @params integer nIndex - Index(0-basiert) des gewhlten Klemmsystems in der Klemmsystemliste
 * @returns	object - aktuell gewhltes Klemmsystem
 */
Cascade.Klemmsystem.holeKlemmsystemAusKlemmsystemliste = function( nIndex )
{
	var strBaseStorePath = Cascade.Klemmsystem._strTempStorePath;

	// teste, ob berhaupt schon eine Suchliste zwischengespeichert wurde
	if ( !checkexist( strBaseStorePath, "" ) )
		return( null );
		
	var nCount = getstring( "COUNT", strBaseStorePath ) - 0;
	if ( nIndex >= nCount )	// Suchliste der letzten Suche war leer, oder Index passt nicht mehr zur aktuellen Suchliste
		return( null );
	
	// rekonstruiere Klemmsystem mit angegebenem Index
	var strItemPath = strBaseStorePath + ".RESULT_LIST[" + nIndex + "]";
	var ks = new Klemmsystem()
	ks.serializeIn( strItemPath );
	
	return( ks );
}


/**
 * berechne den Preis fr ein Klemmsystem
 *
 * @params object Klemmsystem - Klemmsystem, fr das der Preis berechnet werden soll
 * @returns	string der Preis des Klemmsystems
 */
Cascade.Klemmsystem.holePreis = function( Klemmsystem )
{
	var strPreis = '';
	
	// baue select-statement
	var strSelect = "SELECT PREIS FROM KXX WHERE TYP = '" + Klemmsystem.strBasistyp + 
					"' AND PROFILFORM = '" + Klemmsystem.strProfilform + "' AND TRAEGERBREITE = '" + Klemmsystem.strDBTraegerbreite + 
					"' AND WERK_ = '" + Klemmsystem.strWerk + "' ";//AND OBFL like '%" + Oberflaechenschutz[ Klemmsystem.m_Poberfl ] + "%' ";
	var strDBCache = "RH_PREIS";
	if ( dbselect( "LAGER", strSelect, strDBCache ) )
		strPreis = ( dbgetstring( strDBCache, "PREIS" ) - 0 ).toFixed( 2 );
		
	return( strPreis );
}
