/**
 * @fileoverview Base Class for EasyGUI Widgets
 *
 * @author	Gaetan Lauff	<glauff@plansoft.de>
 * @package	js.widgets
 *
 * @version $Id: class.widget.js 248 2007-02-27 10:57:20Z glauff $
 */

var EasyGUI_widget = Class.create();

EasyGUI_widget.prototype = 
{
	/**
	 * @var	boolean
	 */
	initialized: false,
	
	/**
	 * @var object
	 */
	element: null,
	
	/**
	 * @var array
	 */
	widgetElements: [],
	
	/**
	 * @var object
	 */
	widget: null,
	
	/**
	 * @var boolean
	 */
	isInTable: false,
	
	/**
	 * @private
	 */
	initialize: function( id )
	{
		this.id = id;
		this.element = $(id);
		this.isInTable = ( Element.isInTable( this.id ) ) ? true : false;
	},
	
	/**
	 * Sets a custom Action to a component
	 * Only needed in Web Application Frameworks
	 * @access public
	 */
	setAction: function( action ) 
	{
		this.element.setAttribute( 'action', action );
	},
	
	/**
	 * @abstract
	 */
	render: function() 
	{
		;
	},
	
	/**
	 * @abstract
	 */
	update: function()
	{
		;
	},
	
	/**
	 * @abstract
	 */
	finalize: function() 
	{
		this.initialized = true;
	},
	
	/**
	 * Destroys the widget (Circular References in IE )
	 * @private
	 */
	destroy: function()
	{
		this.element = null;
		this.widget = null;
	},
	
	/**
	 * @returns Class Name of DOM Element
	 * @access public
	 */
	getClassName: function()
	{
		return this.element.className;
	},
	
	/**
	 * @returns Style Definition
	 * @access	public
	 */
	getStyleDefinition: function( def )
	{
		return this.element.className + '-' + def;
	},
	
	/**
	 * Used to get the type of object
	 * @returns	string
	 * @access	public
	 */
	toString: function() 
	{
		return 'object EasyGUI_Widget(' + this.id + ')';
	},
	
	/**
	 * Get left position of widget
	 * @returns	integer
	 */
	getOffsetLeft: function() 
	{
		return parseInt(this.element.offsetLeft);
	},
	
	/**
	 * Get the cummulated left position of Widget
	 * @returns integer
	 */
	getPageOffsetLeft: function() 
	{
  		return Position.cumulativeOffset( this.element )[0];
	},
	
	/**
	 * Get top position of widget
	 * @returns	integer
	 */
	getOffsetTop: function() 
	{ 
		return parseInt(this.element.offsetTop);
	},
	
	/**
	 * Get top position of widget in page context
	 * @returns integer
	 */
	getPageOffsetTop: function() 
	{
		return Position.cumulativeOffset( this.element )[1];
	},
	
	/**
	 * Get width of widget
	 * @returns	integer
	 */
	getOffsetWidth: function() 
	{ 
		return parseInt( this.element.offsetWidth ); 
	},

	/**
	 * Get height of widget
	 * @returns	integer
	 */
	getOffsetHeight: function() 
	{
		return parseInt( this.element.offsetHeight );
	},
	
	/**
	 * Get left style position of widget
	 * @returns	integer
	 */
	getStyleLeft: function() 
	{
		return parseInt( this.element.style.left );
	},
	
	/**
	 * Get top position of widget
	 * @returns	integer
	 */
	getStyleTop: function() 
	{ 
		return parseInt( this.element.style.top );
	},
	
	/**
	 * Get width of widget
	 * @returns	integer
	 */
	getStyleWidth: function() 
	{ 
		return parseInt( this.element.style.width ); 
	},

	/**
	 * Get height of widget
	 * @returns	int
	 */
	getStyleHeight: function() 
	{
		return parseInt( this.element.style.height );
	},
	
	/**
	 * Move the widget by a given amount of pixels to the left
	 *
	 * @param	{int} pixels	Amount of pixels to move widget to the left
	 * @returns	void
	 * @access	public
	 */
	moveLeft: function(pixels) 
	{
		this.element.style.left = ( this.getStyleLeft( ) - parseInt( pixels ) ) + 'px';
	},
	
	/**
	 * Move the widget by a givem amount of pixels to the right
	 *
	 * @param	{int}	pixels	Amount of pixels to move widget right
	 * @returns	void
	 * @access	public
	 */
	moveRight: function(pixels) 
	{
		this.element.style.left = ( this.getStyleLeft( ) + parseInt( pixels ) ) + 'px';
	},
	
	/**
	 * Move the widget by a given amount of pixels to the top
	 *
	 * @param	{int}	pixels	Amount of pixels to move widget up
	 * @returns	void
	 * @access	public
	 */
	moveUp: function(pixels) 
	{
		this.element.style.top = ( this.getStyleTop() - parseInt( pixels ) ) + 'px';
	},
	
	/**
	 * Move the widget by a given amount of pixels down
	 *
	 * @param	{int}	pixels	Amount of pixels to move widget down
	 * @return	void
	 * @access	public
	 */
	moveDown: function(pixels) 
	{
		this.element.style.top = ( this.getStyleTop() + parseInt(pixels)) + 'px';
	},
	
	/**
	 * Sets the position of widget's DOM Element
	 *
	 * @param	int		top		Top position in pixels
	 * @param	int 	left	Left position in pixels
	 * @access	public
	 */
	setPosition: function( top, left ) 
	{
		var top = ( typeof top == 'string' ) ? top : top + 'px';
		var left = ( typeof left == 'string' ) ? left : left + 'px';
		
		this.element.style.top = top;
		this.element.style.left = left;
	},
	
	/**
	 * Sets the dimension of widget's DOM Element
	 *
	 * @param	int		width		width in pixels
	 * @param	int 	height	height in pixels
	 * @access	public
	 */
	setDimension: function( width, height ) 
	{
		var width = ( typeof width == 'string' ) ? width : width + 'px';
		var height = ( typeof height == 'string' ) ? height : height + 'px';
	
		this.element.style.width = width;
		this.element.style.height = height;
	},
	
	/**
	 * Disables widget
	 *
	 * @returns	void
	 * @access	public
	 */
	disable: function() 
	{ 
		this.element.disabled = true;
	},
	
	/**
	 * Enables widget
	 *
	 * @returns	void
	 * @access	public
	 */
	enable: function() 
	{
		this.element.disabled = false;
	},
	
	/**
	 * Returns the Value
	 */
	getValue: function() {
		return $F(this.element);
	},
	
	/**
	 * Sets the Value
	 */
	setValue: function( value ) {
		this.element.value = value;
	},
	
	
	/**
	 * Shows Widget
	 *
	 * @returns	void
	 * @acces	public
	 */
	show: function() 
	{
		this.element.style.visibility = 'visible';
	},
	
	/**
	 * Hides Widget
	 *
	 * @returns	void
	 * @access	public
	 */
	hide: function()
	{
		this.element.style.visibility = 'hidden';
	}
};