/**
 * @fileoverview	Library for handling dates and times
 *
 * @author	Gaetan Lauff	<glauff@plansoft.de>
 * @version $Id: lib.datetime.js 248 2007-02-27 10:57:20Z glauff $
 */
var DateTime = {
	/**
	 * Checks if a date lies in the future
	 *
	 * @param	{int}	dateStr
	 * @param	{int}	format
	 *
	 * @returns	TRUE if the given date lies in the future, FALSE otherwise
	 *
	 * @access	public
	 */
	isFutureDate: function( dateStr, format)
	{
		var parts 	= DateTime.getParts( dateStr, format );
		
		var day		= DateTime.getPart(parts, 'd', format);
		var month	= DateTime.getPart(parts, 'm', format);
		var year	= DateTime.getPart(parts, 'y', format);
		
		var today = new Date();
		
		if (parseFloat(year) <= 50) {
			year = (parseFloat(year) + 2000).toString(); 
		}
		
		if (parseFloat(year) <= 99) {
			year = (parseFloat(year) + 1900).toString();
		}
		
		var dt = new Date(parseFloat(year), parseFloat(month), parseFloat(day), 0, 0, 0, 0);
		
		var tim1 = dt.getTime();
		var tim2 = today.getTime();
		
		if (tim1 > tim2) {
			return true;
		}
		
		return false;
	},
	
	/**
	 * Checks if a date is valid
	 *
	 * @param	{int}	dateStr
	 * @param	{int}	format
	 *
	 * @returns	TRUE if the given date is valid, FALSE otherwise.
	 *
	 * @access	public
	 */
	isValidDate: function( dateStr, format )
	{
		var parts 	= DateTime.getParts( dateStr, format );
		
		if ( !parts) {
			return false;
		}
		
		var day		= DateTime.getPart(parts, 'd', format);
		var month	= DateTime.getPart(parts, 'm', format);
		var year	= DateTime.getPart(parts, 'y', format);
		
		if (parseFloat(year) <= 50) {
			year = (parseFloat(year) + 2000).toString(); 
		}
		
		if (parseFloat(year) <= 99) {
			year = (parseFloat(year) + 1900).toString();
		}
		
		var dt = new Date(parseFloat(year), parseFloat(month), parseFloat(day), 0, 0, 0, 0);
		
		if (parseFloat(day) != dt.getDate()) {
			return false;
		}
		
		if (parseFloat(month) != dt.getMonth()) {
			return false; 
		}
		
		return true;
	},
	
	/**
	 * Checks the given format
	 * @param	{string}	format
	 * @returns	The right date format lowercased
	 * @access	public
	 */
	getFormat: function( format )
	{
		if (format == null) {
			format = "dmy";
		}
		
		format = format.toLowerCase();
		
		if (format.length != 3) {
	   		format = "dmy";
		}
		   
	   	if ( (format.indexOf("m") == -1) || (format.indexOf("d") == -1) || (format.indexOf("y") == -1) ) {
	      	format = "dmy";
	   	}
	   	
	   	return format;
	},
	
	/**
	 * This function is used to get an array of date parts from a given date String according to a specific format
	 * 
	 * @param	{string}	dateStr
	 * @param	{string}	format
	 */
	getParts: function(dateStr, format)
	{
		var format = DateTime.getFormat( format );
		
	   	if (format.substring(0, 1) == "y") { // If the year is first
	    	var reg1 = /^\d{2}(\-|\/|\.)\d{1,2}\1\d{1,2}$/;
	      	var reg2 = /^\d{4}(\-|\/|\.)\d{1,2}\1\d{1,2}$/;
	   	} else if (format.substring(1, 2) == "y") { // If the year is second
	    	var reg1 = /^\d{1,2}(\-|\/|\.)\d{2}\1\d{1,2}$/;
	      	var reg2 = /^\d{1,2}(\-|\/|\.)\d{4}\1\d{1,2}$/;
	   	} else { // The year must be third
	    	var reg1 = /^\d{1,2}(\-|\/|\.)\d{1,2}\1\d{2}$/;
	      	var reg2 = /^\d{1,2}(\-|\/|\.)\d{1,2}\1\d{4}$/;
	   	}
	
	   	if ( (reg1.test(dateStr) == false) && (reg2.test(dateStr) == false) ) {
	   		return false; 
	   	}
	   
	   	var parts = dateStr.split(RegExp.$1); // Split into 3 parts based on what the divider was
	   	return parts;
	},
	
	/**
	 * This function is used to extract a specific part from the date parts array
	 * 
	 * @param	{array}		parts	Date Parts
	 * @param	{string}	part	'd' for day, 'm' for month and 'y' for year.
	 * @param	{string}	format
	 *
	 * @returns	The value from the date parts array according to the given format.
	 *
	 * @access	public
	 */
	getPart: function(parts, part, format)
	{
		if (part == null) {
			part = 'y';
		}
			
		var part = part.toLowerCase();
		
		if (part != 'm' && part != 'y' && part != 'd') {
			return false;
		}
			
		var format = DateTime.getFormat( format );
		
	   	if (format.substring(0, 1) == part) {
	   		var result = parts[0]; 
	   	} else if (format.substring(1, 2) == part) {
	   		var result = parts[1];
	   	} else {
	   		var result = parts[2];
	   	}
	   	
	   	if ( part == 'm' ) {
	   		return result - 1;
	   	}
	   	
	   	return result;
	},
	
	/**
	 * Checks if passed argument is a valid hour
	 * @param	{string}	hour
	 * @returns	TRUE if the given hour is a valid hour, FALSE otherwise.
	 */
	isValidTime: function(hour)
	{
		return /^([01]?[0-9]|[2][0-3])(:[0-5][0-9])?$/.test(hour);
	}
};